<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    public function profile()
    {
        $pageTitle = "Profile Setting";
        $user = auth()->user();
        return view('Template::user.profile_setting', compact('pageTitle','user'));
    }

    public function submitProfile(Request $request)
    {
        $request->validate([
            'firstname' => 'required|string',
            'lastname' => 'required|string',
        ],[
            'firstname.required'=>'The first name field is required',
            'lastname.required'=>'The last name field is required'
        ]);

        $user = auth()->user();

        $user->firstname = $request->firstname;
        $user->lastname = $request->lastname;

        $user->address = $request->address;
        $user->city = $request->city;
        $user->state = $request->state;
        $user->zip = $request->zip;

        $user->save();
        $notify[] = ['success', 'Profile updated successfully'];
        return back()->withNotify($notify);
    }
    
    public function profileUpload(Request $request)
{
    $request->validate([
        'image' => 'required|image|mimes:jpg,jpeg,png|max:2048',
    ], [
        'image.required' => 'Please select an image to upload',
        'image.image'    => 'The file must be an image',
        'image.mimes'    => 'Only JPG, JPEG, and PNG files are allowed',
        'image.max'      => 'Image size must not exceed 2MB',
    ]);

    $user = auth()->user();
    $filename = $user->image; // Keep old image by default

    if ($request->hasFile('image')) {
        try {
            $filename = fileUploader(
                $request->image,
                getFilePath('userProfile'),
                old: $filename
            );
        } catch (\Exception $exp) {
            $notify[] = ['error', 'Profile image could not be uploaded'];
            return back()->withNotify($notify);
        }
    }

    $user->image = $filename;
    $user->save();

    $notify[] = ['success', 'Profile image uploaded successfully'];
    return back()->withNotify($notify);
}

    
    public function transactionPassword()
    {
        $pageTitle = 'Transaction PIN';
        $user = auth()->user();
        return view('Template::user.pin', compact('pageTitle','user'));
    }

    public function trxPassword(Request $request)
    {
        $passwordValidation = Password::min(6);
        if (gs('secure_password')) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

         $request->validate([
            'password' => 'required',
        ],[
            'password.required'=>'Please provide transaction Pin'
        ]);

        $user = auth()->user();
        
            $password       = Hash::make($request->password);
            $user->transaction_password = $password;
            $user->save();
            $notify[] = ['success', 'Password changed successfully'];
            return back()->withNotify($notify);
        
    }
    
    
    public function submitMobile(Request $request)
    {
        $request->validate([
            'mobile' => 'required',
        ],[
            'mobile.required'=>'Please provide phone number'
        ]);

        $user = auth()->user();

        $user->mobile = $request->mobile;

        $user->save();
        $notify[] = ['success', 'Phone Number updated successfully'];
        return back()->withNotify($notify);
    }
    
    public function submitEmail(Request $request)
    {
        $request->validate([
            'email' => 'required|unique:users',
        ],[
            'email.required'=>'Please provide email address',
            'email.unique'=>'This email is associated with another account'
        ]);

        $user = auth()->user();

        $user->email = $request->email;

        $user->save();
        $notify[] = ['success', 'Email updated successfully'];
        return back()->withNotify($notify);
    }

    public function changePassword()
    {
        $pageTitle = 'Change Password';
        return view('Template::user.password', compact('pageTitle'));
    }

    public function submitPassword(Request $request)
    {

        $passwordValidation = Password::min(6);
        if (gs('secure_password')) {
            $passwordValidation = $passwordValidation->mixedCase()->numbers()->symbols()->uncompromised();
        }

        $request->validate([
            'current_password' => 'required',
            'password' => ['required','confirmed',$passwordValidation]
        ]);

        $user = auth()->user();
        if (Hash::check($request->current_password, $user->password)) {
            $password = Hash::make($request->password);
            $user->password = $password;
            $user->save();
            $notify[] = ['success', 'Password changed successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'The password doesn\'t match!'];
            return back()->withNotify($notify);
        }
    }
}
